/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */
package djbuch.kapitel_19;

import java.io.File;
import java.util.Vector;

import djbuch.kapitel_06.DemoTools;
import djbuch.kapitel_06.GC;

import lotus.domino.Database;
import lotus.domino.DateTime;
import lotus.domino.Document;
import lotus.domino.EmbeddedObject;
import lotus.domino.Item;
import lotus.domino.NotesException;
import lotus.domino.RichTextItem;
import lotus.domino.Session;

//TSA*JAVA0174
/**
* 
* Implementiert eine Email, die ber die Notes Mail.box verschickt wird.
* Der besondere Vorteil an dieser Klasse ist, dass die Mail Absender frei definiert
* werden knnen.
* 
* @author Thomas Ekert
*/
public class EmailNotes extends Message {

	private String replyTo = null;
	private String subject = null;
	private String [] recipientArr = null;
	private Session session = null;
	private Database mailBox = null;
	private String attachFileName = null;
	private String serverName = "";

	/**
	 * @param type
	 * @param configuration
	 * @throws MessageException
	 */
	public EmailNotes(Session s) throws MessageException {
		super (Message.TYPE_EMAIL, null);
		this.session = s;
		try {
			this.serverName = s.getServerName();
		} catch (NotesException e) {
			throw new MessageException ("Konnte den Servernamen nicht erhalten.");
		}
	}
	
	/**
	 * @param type
	 * @param configuration
	 * @throws MessageException
	 */
	public EmailNotes(Session s, String serverName) throws MessageException {
		super (Message.TYPE_EMAIL, null);
		this.session = s;
		this.serverName = serverName;
	}
	
	/**
	 * rumt die NotesRessourcen auf.
	 */
	public void recycle () {
		mailBox = GC.recycle (mailBox);
	}
	/**
	 * wird dieser Wert gesetzt, wird der Email eine Datei diesen Namens angehngt
	 * und nach Versand GELSCHT!!
	 * @param fn
	 */
	public void setAttachFileName (String fn) {
		attachFileName = fn;
	}
	
	/**
	 * Setzt den Absender der E-Mail
	 * Wirft eine MessageException, falls der Absender nicht ok ist.
	 */
	public void setSender(String sender) throws MessageException {
		if (isRecipientOK(sender)) {
			this.sender = sender;
			return;
		} 
		throw new MessageException ("Unzulaessiger Absender.");
	}

	/**
	 * berprft, ob der Empfnger eine korrekte Email ist.
	 */
	protected boolean isRecipientOK(String rec) {
		if (DemoTools.isEmpty (rec)) {
			return false;
		}
		/* Ben : Check ob es sich um eine Notesmail handelt. sollte etwas verbessert werden. */
		if (rec.indexOf("/") != -1) {
			return true;
		}
		return validate_email(rec);
	}
	
	/** Bei der Verwendung mehrer Empfnger: 
	 * berprfung ob alle ok sind.
	 * @param rec - Array von Strings
	 * @return - true, falls alle ok, false, falls mindestens einer nicht ok.
	 */
	public boolean isRecipientOK (String [] rec) {
		if (rec == null || rec.length == 0) {
			return false;
		}
		for (int i = 0; i < rec.length; i++) {
			if (! isRecipientOK(rec[i])) {
				return false;
			}
		}
		return true;
	}

	/**
	 * @deprecated - Diese Methode darf fr diese Implementierung
	 * von Message nicht verwendet werden.
	 * @see getRecipientArray ()
	 */
	public String getRecipient ()  {
		System.err.println ("Verwenden Sie getRecipientArray.");
		return null;
	}

	/** liefert das Array der Empfnger
	 * 
	 * @return
	 */
	public String [] getRecipientArray () {
		return this.recipientArr;
	}
	
	/** Setzt einen einzelnen Empfnger
	 * 
	 */
	public void setRecipient (String rec) throws MessageException {
		String [] recArr = new String [1];
		recArr [0] = rec;
		this.setRecipient(recArr);
	}
	
	/**
	 * Setzt den ReplyTo sofern nicht null
	 * Falls nicht null, muss es sich um eine korrekte E-Mail handeln.
	 * @param replyTo
	 * @throws MessageException
	 */
	public void setReplyTo (String replyTo) throws MessageException {
		if (replyTo == null) {
			this.replyTo = null;
			return;
		}
		if (isRecipientOK(replyTo)) {
			this.replyTo = replyTo;
		} else {
			throw new MessageException ("ReplyTo war nicht ok.");
		}
	}
	
	/**
	 * gibt den ReplyTo (String) zurck.
	 * @return
	 */
	public String getReplyTo () {
		return this.replyTo;
	}

	/**
	 * Setzt mehrere Empfnger fr die Message. 
	 * @param rec - Empfnger
	 * @throws MessageException - falls der zu Setzende Recipient nicht ok (hierfr muss isRecipientOK implementiert werden.) ist.
	 */
	public void setRecipient (String [] rec) throws MessageException{
		if (rec == null) {
			throw new MessageException ("Das Array aus Empfaengern darf nicht null sein.");
		}
		if (isRecipientOK(rec)) {
			this.recipientArr = rec;
		} else {
			throw new MessageException ("Einer der Empfaenger war nicht ok.");
		}
	}

	/**
	 * Setzt die Nachricht er E-Mail
	 */
	public void setMessage(String message) throws MessageException {
		if (message == null) {
			throw new MessageException ("Die Nachricht darf nicht null sein.");
		}
		this.message = message;
	}

	/** 
	 * legt das Subject der Email fest.
	 * @param sub
	 * @throws MessageException
	 */
	public void setSubject (String sub) throws MessageException {
		if (sub == null) {
			throw new MessageException ("Das Subject darf nicht null sein.");
		}
		this.subject = sub;
	}
	
	/**
	 * @return - das Subject der email
	 */
	public String getSubject () {
		return this.subject;
	}

	/**
	 * versendet die Email ber die Notes mail.box
	 */
	public boolean send() throws MessageException {
		if (!isRecipientOK(getRecipientArray())) {
			System.err.println ("Sie muessen zuvor einen Empfnger festlegen.");
			return false;
		}
		if (!isRecipientOK (getSender ())) {
			System.err.println ("Sie muessen zuvor einen Absender festlegen.");
			return false;
		}
		if (DemoTools.isEmpty(this.subject)) {
			System.err.println ("Sie muessen zuvor ein Subject festlegen.");
			return false;
		}
		if (DemoTools.isEmpty(getMessage())) {
			System.err.println ("Sie muessen zuvor einen Mail Body festlegen.");
			return false;
		}
		this.sendStatus = false;
		this.sendCount = 0;
		Vector vSubject = new Vector ();
		vSubject.add(this.subject);
		for (int i = 0; i < getRecipientArray().length; i++) {
			if (sendViaMailBox(getRecipientArray()[i])) {
				this.sendCount++;
			} 
		}
		if (this.sendCount == getRecipientArray().length) {
			this.sendStatus = true;
		}
		return this.sendStatus;
	}
	

	/**
	 * verschickt den Text aus bodyTxt als Mail ber die MAIL.BOX-Datenbank
	 */
	private boolean sendViaMailBox(String recipient)
		throws MessageException {
		Vector recipientVector = new Vector();
		recipientVector.add(recipient);
		Document doc = null;
		Item sendToItem = null;
		Item recipientItem = null;
		Item blindCopyItem = null;
		boolean allesOK = false;
		try {
			if (getMailBox() == null) {
				return false;
			}
			doc = getMailBox().createDocument();
			recipientItem = doc.replaceItemValue("Recipients", recipientVector);
			recipientItem.setNames(true);
			sendToItem = doc.replaceItemValue("SendTo", recipientVector);
			sendToItem.setNames(true);
			if (getReplyTo() != null && ! getReplyTo().equalsIgnoreCase(getSender())) {
				doc.replaceItemValue("replyTo", getReplyTo());
			}
			// Blind Copy - nicht implementiert ...
			blindCopyItem = doc.replaceItemValue("BlindCopyTo", new Vector ());
			blindCopyItem.setNames(true);
			
			recipientItem.setSummary(true);
			sendToItem.setSummary(true);
			
			doc.replaceItemValue("Subject", getSubject());
			doc.replaceItemValue("From", getSender());
			
			// Die Mail Nachricht gehrt ins Body Item
			RichTextItem body = doc.createRichTextItem("Body");
			body.appendText(getMessage());
			// und auch das Attachment
			if (attachFileName != null) {
				File f = new File (attachFileName);
				if (f.exists()) {
					//System.out.println ("Attaching to Email: " + f.getAbsolutePath());
					body.addNewLine(2);
					body.embedObject(EmbeddedObject.EMBED_ATTACHMENT,"",f.getAbsolutePath(),"Dateianhang");
					body.addNewLine(2);
				} else {
					System.err.println ("Fehler: Datei " + f.getAbsolutePath() + " konnte nicht gefunden werden.");
				}
			}

			// Ein Document in der mail.box bentigt einige zustzliche Items...
			doc.replaceItemValue("Form", "Memo");
			doc.replaceItemValue("DeliveryPriority", "High");
			doc.replaceItemValue("MailFormat", "T");
			doc.replaceItemValue("deliveryReport", "O");
			DateTime dt = session.createDateTime("Today");
			dt.setNow();
			doc.replaceItemValue("PostedDate",dt);
			allesOK = doc.save(true, false);
		} catch (Exception e) {
			throw new MessageException(e);
		} finally {
			try {
				recipientItem = GC.recycle (recipientItem);
				sendToItem = GC.recycle(sendToItem);
				blindCopyItem = GC.recycle(blindCopyItem);
				doc = GC.recycle(doc);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return allesOK;
	}

	private Database getMailBox () throws NotesException {
		if (mailBox == null || !mailBox.isOpen()) {
			try {
				mailBox = session.getDatabase(serverName,"mail.box");
			} catch (NotesException e) {
				mailBox = null;
			}
			if ((mailBox == null) || (!mailBox.isOpen())) {
				System.err.println ("Konnte mail.box nicht ffnen.");
			}
		}
		return mailBox;
	}
	
	/** oberflchliche Prfung der Email-Adresse,
	*    hauptschlich dass es genau 1 @-Zeichen darin gibt
	*    und der Domainpart mindestens einen Punkt enthlt
	*/
	static boolean validate_email(String s) {
		try {
			int index_At = s.indexOf('@');
			if (index_At < 1) {
				return false;
			}
			String domain = s.substring(index_At + 1);
			if (domain.indexOf('@') >= 0) {
				return false;
			}
			int index_toplevel_dom = domain.lastIndexOf('.');
			if (index_toplevel_dom < 0) {
				return false;
			}
			int toplevel_len = domain.length() - index_toplevel_dom - 1;
			return (toplevel_len >= 1 && toplevel_len <= 4);
		} catch (StringIndexOutOfBoundsException e) {
			System.err.println("validate_email(" + s + "):" + e);
			return false;
		}
	}
}
